#' **bayesplot**: Plotting for Bayesian Models
#'
#' @name bayesplot-package
#' @aliases bayesplot
#'
#' @import ggplot2 stats rlang
#' @importFrom dplyr %>% summarise group_by select
#'
#' @description
#' \if{html}{
#'    \figure{logo.svg}{options: width="50" alt="mc-stan.org"}
#' }
#' *Stan Development Team*
#'
#' The **bayesplot** package provides a variety of **ggplot2**-based
#' plotting functions for use after fitting Bayesian models (typically, though
#' not exclusively, via Markov chain Monte Carlo). The package is designed not
#' only to provide convenient functionality for users, but also a common set of
#' functions that can be easily used by developers working on a variety of
#' packages for Bayesian modeling, particularly (but not necessarily) packages
#' powered by RStan (the \R interface to Stan).
#' Examples of packages that will soon (or already are) using **bayesplot** are
#' **rstan** itself, as well as the **rstan**-dependent **rstanarm** and
#' **brms** packages for applied regression modeling.
#'
#' @section Plotting functionality:
#'  \if{html}{
#'    \figure{bayesplot1.png}{options: style="width: 30\%;" alt="mcmc_areas"}
#'    \figure{bayesplot2.png}{options: style="width: 30\%;" alt="ppc_hist"}
#'    \figure{bayesplot3.png}{options: style="width: 30\%;" alt="ppc_dens_overlay"}
#'  }
#'
#' The plotting functions in **bayesplot** are organized into several modules:
#' * [MCMC][MCMC-overview]: Visualizations of Markov chain
#'   Monte Carlo (MCMC) simulations generated by *any* MCMC algorithm
#'   as well as diagnostics. There are also additional functions specifically
#'   for use with models fit using the [No-U-Turn Sampler (NUTS)][NUTS].
#' * [PPC][PPC-overview]: Graphical (posterior or prior) predictive
#'   checks (PPCs).
#' * [PPD][PPD-overview]: Plots of (posterior or prior) predictive
#'   distributions without comparisons to observed data.
#'
#' @section Resources:
#' * __Online documentation and vignettes__: Visit the __bayesplot__ website at
#'   <https://mc-stan.org/bayesplot/>
#' * __Bug reports and feature requests__: If you would like to request a new
#'   feature or if you have noticed a bug that needs to be fixed please let us
#'   know at the **bayesplot** issue tracker at
#'   <https://github.com/stan-dev/bayesplot/issues/>
#' * __General questions and help__:
#'   To ask a question about **bayesplot** on the Stan Forums forum please visit
#'   <https://discourse.mc-stan.org>.
#'
#' @template seealso-theme
#' @template seealso-colors
#' @seealso [ggplot2::ggsave()] for saving plots.
#'
#' @template reference-vis-paper
#'
#' @examples
#' # A few quick examples (all of the functions have many examples
#' # on their individual help pages)
#'
#' # MCMC plots
#' x <- example_mcmc_draws(params = 5)
#' mcmc_intervals(x, prob = 0.5)
#' mcmc_intervals(x, regex_pars = "beta")
#'
#' color_scheme_set("purple")
#' mcmc_areas(x, regex_pars = "beta", prob = 0.8)
#'
#' color_scheme_set("mix-blue-red")
#' mcmc_trace(x, pars = c("alpha", "sigma"),
#'            facet_args = list(nrow = 2))
#'
#' color_scheme_set("brightblue")
#' mcmc_scatter(x, pars = c("beta[1]", "sigma"),
#'              transformations = list(sigma = "log"))
#'
#'
#' # Graphical PPCs
#' y <- example_y_data()
#' yrep <- example_yrep_draws()
#' ppc_dens_overlay(y, yrep[1:50, ])
#' \donttest{
#' color_scheme_set("pink")
#' ppc_stat(y, yrep, stat = "median") + grid_lines()
#' ppc_hist(y, yrep[1:8, ])
#'
#' # Same plots but without y (using ppd_ instead of ppc_)
#' bayesplot_theme_set(ggplot2::theme_gray())
#' ypred <- yrep
#' ppd_dens_overlay(ypred[1:50, ])
#' ppd_stat(ypred, stat = "median") + grid_lines()
#' ppd_hist(ypred[1:8, ])
#' }
#'
"_PACKAGE"


# internal ----------------------------------------------------------------

# nocov start
# release reminders (for devtools)
release_questions <- function() {
  c(
    "Have you tested with '^tests/testthat/_snaps$' removed from .Rbuildignore?",
    "Have you put '^tests/testthat/_snaps$' back in .Rbuildignore? before submitting?"
  )
}
# nocov end
