% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grasps.R
\name{grasps}
\alias{grasps}
\title{Groupwise Regularized Adaptive Sparse Precision Solution}
\usage{
grasps(
  X,
  n = nrow(X),
  membership,
  penalty,
  diag.ind = TRUE,
  diag.grp = TRUE,
  diag.include = FALSE,
  lambda = NULL,
  alpha = NULL,
  gamma = NULL,
  nlambda = 10,
  lambda.min.ratio = 0.01,
  growiter.lambda = 30,
  tol.lambda = 0.001,
  maxiter.lambda = 50,
  rho = 2,
  tau.incr = 2,
  tau.decr = 2,
  nu = 10,
  tol.abs = 1e-04,
  tol.rel = 1e-04,
  maxiter = 10000,
  crit = "BIC",
  kfold = 5,
  ebic.tuning = 0.5
)
}
\arguments{
\item{X}{\enumerate{
\item An \eqn{n \times d} data matrix with sample size \eqn{n} and
dimension \eqn{d}.
\item A \eqn{d \times d} sample covariance matrix with dimension \eqn{d}.
}}

\item{n}{An integer (default = \code{nrow(X)}) specifying the sample size.
This is only required when the input matrix \code{X} is a \eqn{d \times d}
sample covariance matrix with dimension \eqn{d}.}

\item{membership}{An integer vector specifying the group membership.
The length of \code{membership} must be consistent with the dimension \eqn{d}.}

\item{penalty}{A character string specifying the penalty for estimating
precision matrix. Available options include:
\enumerate{
\item "lasso": Least absolute shrinkage and selection operator
\insertCite{tibshirani1996regression,friedman2008sparse}{grasps}.
\item "adapt": Adaptive lasso
\insertCite{zou2006adaptive,fan2009network}{grasps}.
\item "atan": Arctangent type penalty
\insertCite{wang2016variable}{grasps}.
\item "exp": Exponential type penalty
\insertCite{wang2018variable}{grasps}.
\item "lq": Lq penalty
\insertCite{frank1993statistical,fu1998penalized,fan2001variable}{grasps}.
\item "lsp": Log-sum penalty
\insertCite{candes2008enhancing}{grasps}.
\item "mcp": Minimax concave penalty
\insertCite{zhang2010nearly}{grasps}.
\item "scad": Smoothly clipped absolute deviation
\insertCite{fan2001variable,fan2009network}{grasps}.
}}

\item{diag.ind}{A logical value (default = TRUE) specifying whether to
penalize the diagonal elements.}

\item{diag.grp}{A logical value (default = TRUE) specifying whether to
penalize the within-group blocks.}

\item{diag.include}{A logical value (default = FALSE) specifying whether to
include the diagonal entries in the penalty for within-group blocks when
\code{diag.grp = TRUE}.}

\item{lambda}{A non-negative numeric vector specifying the grid for
the regularization parameter. The default is \code{NULL}, which generates
its own \code{lambda} sequence based on \code{nlambda} and
\code{lambda.min.ratio}.}

\item{alpha}{A numeric vector in [0, 1] specifying the grid for
the mixing parameter balancing the element-wise individual L1 penalty and
the block-wise group L2 penalty.
An alpha of 1 corresponds to the individual penalty only; an alpha of 0
corresponds to the group penalty only.
The default value is a sequence from 0.1 to 0.9 with increments of 0.1.}

\item{gamma}{A numeric value specifying the additional parameter fo
the chosen \code{penalty}. The default value depends on the penalty:
\enumerate{
\item "adapt": 0.5
\item "atan": 0.005
\item "exp": 0.01
\item "lq": 0.5
\item "lsp": 0.1
\item "mcp": 3
\item "scad": 3.7
}}

\item{nlambda}{An integer (default = 10) specifying the number of
\code{lambda} values to generate when \code{lambda = NULL}.}

\item{lambda.min.ratio}{A numeric value > 0 (default = 0.01) specifying
the fraction of the maximum \code{lambda} value \eqn{\lambda_{max}} to
generate the minimum \code{lambda} \eqn{\lambda_{min}}.
If \code{lambda = NULL}, a \code{lambda} grid of length \code{nlambda} is
automatically generated on a log scale, ranging from \eqn{\lambda_{max}}
down to \eqn{\lambda_{min}}.}

\item{growiter.lambda}{An integer (default = 30) specifying the maximum
number of exponential growth steps during the initial search for an
admissible upper bound \eqn{\lambda_{\max}}.}

\item{tol.lambda}{A numeric value > 0 (default = 1e-03) specifying
the relative tolerance for the bisection stopping rule on the interval width.}

\item{maxiter.lambda}{An integer (default = 50) specifying the maximum number
of bisection iterations in the line search for  \eqn{\lambda_{\max}}.}

\item{rho}{A numeric value > 0 (default = 2) specifying the ADMM
augmented-Lagrangian penalty parameter (often called the ADMM step size).
Larger values typically put more weight on enforcing the consensus
constraints at each iteration; smaller values yield more conservative updates.}

\item{tau.incr}{A numeric value > 1 (default = 2) specifying
the multiplicative factor used to increase \code{rho} when the primal
residual dominates the dual residual in ADMM.}

\item{tau.decr}{A numeric value > 1 (default = 2) specifying
the multiplicative factor used to decrease \code{rho} when the dual residual
dominates the primal residual in ADMM.}

\item{nu}{A numeric value > 1 (default = 10) controlling how aggressively
\code{rho} is rescaled in the adaptive-\code{rho} scheme (residual balancing).}

\item{tol.abs}{A numeric value > 0 (default = 1e-04) specifying the absolute
tolerance for ADMM stopping (applied to primal/dual residual norms).}

\item{tol.rel}{A numeric value > 0 (default = 1e-04) specifying the relative
tolerance for ADMM stopping (applied to primal/dual residual norms).}

\item{maxiter}{An integer (default = 1e+04) specifying the maximum number of
ADMM iterations.}

\item{crit}{A character string (default = "BIC") specifying the parameter
selection criterion to use. Available options include:
\enumerate{
\item "AIC": Akaike information criterion
\insertCite{akaike1973information}{grasps}.
\item "BIC": Bayesian information criterion
\insertCite{schwarz1978estimating}{grasps}.
\item "EBIC": extended Bayesian information criterion
\insertCite{chen2008extended,foygel2010extended}{grasps}.
\item "HBIC": high dimensional Bayesian information criterion
\insertCite{wang2013calibrating,fan2017high}{grasps}.
\item "CV": k-fold cross validation with negative log-likelihood loss.
}}

\item{kfold}{An integer (default = 5) specifying the number of folds used for
\code{crit = "CV"}.}

\item{ebic.tuning}{A numeric value in [0, 1] (default = 0.5) specifying
the tuning parameter to calculate for \code{crit = "EBIC"}.}
}
\value{
An object with S3 class \code{"grasps"} containing the following components:
\describe{
\item{hatOmega}{The estimated precision matrix.}
\item{lambda}{The optimal regularization parameter.}
\item{alpha}{The optimal mixing parameter.}
\item{initial}{The initial estimate of \code{hatOmega} when a non-convex
penalty is chosen via \code{penalty}.}
\item{gamma}{The optimal addtional parameter when a non-convex penalty
is chosen via \code{penalty}.}
\item{iterations}{The number of ADMM iterations.}
\item{lambda.grid}{The actual lambda grid used in the program.}
\item{alpha.grid}{The actual alpha grid used in the program.}
\item{lambda.safe}{The bisection-refined upper bound \eqn{\lambda_{\max}},
corresponding to \code{alpha.grid}, when \code{lambda = NULL}.}
\item{loss}{The optimal k-fold loss when \code{crit = "CV"}.}
\item{CV.loss}{Matrix of CV losses, with rows for parameter combinations and
columns for CV folds, when \code{crit = "CV"}.}
\item{score}{The optimal information criterion score when \code{crit} is set
to \code{"AIC"}, \code{"BIC"}, \code{"EBIC"}, or \code{"HBIC"}.}
\item{IC.score}{The information criterion score for each parameter
combination when \code{crit} is set to \code{"AIC"}, \code{"BIC"},
\code{"EBIC"}, or \code{"HBIC"}.}
\item{membership}{The group membership.}
}
}
\description{
Provide a collection of statistical methods that incorporate both
element-wise and group-wise penalties to estimate a precision matrix.
}
\examples{
library(grasps)

## reproducibility for everything
set.seed(1234)

## block-structured precision matrix based on SBM
sim <- gen_prec_sbm(d = 30, K = 3,
                    within.prob = 0.25, between.prob = 0.05,
                    weight.dists = list("gamma", "unif"),
                    weight.paras = list(c(shape = 20, rate = 10),
                                        c(min = 0, max = 5)),
                    cond.target = 100)
## visualization
plot(sim)

## n-by-d data matrix
library(MASS)
X <- mvrnorm(n = 20, mu = rep(0, 30), Sigma = sim$Sigma)

## adapt, HBIC
res <- grasps(X = X, membership = sim$membership, penalty = "adapt", crit = "HBIC")

## visualization
plot(res)

## performance
performance(hatOmega = res$hatOmega, Omega = sim$Omega)
}
\references{
\insertAllCited{}
}
