% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spec_area_targets.R
\name{spec_area_targets}
\alias{spec_area_targets}
\title{Specify targets based on area units}
\usage{
spec_area_targets(targets, area_units)
}
\arguments{
\item{targets}{\code{numeric} vector denoting the
area-based targets for the features.
These values must be expressed in the same units as \code{area_units}.
If a single \code{numeric} value is specified, then all
features are assigned targets based on the area-based target.}

\item{area_units}{\code{character} vector denoting the unit of measurement
for each \code{targets} value (e.g., \code{"km^2"}, \code{"ha"}, and \code{"acres"}).
If a single \code{character} value is specified, then all
features are assigned targets assuming that \code{targets}
are in the same units.}
}
\value{
An object (\code{\linkS4class{TargetMethod}}) for specifying targets that
can be used with \code{\link[=add_auto_targets]{add_auto_targets()}} and \code{\link[=add_group_targets]{add_group_targets()}}
to add targets to a \code{\link[=problem]{problem()}}.
}
\description{
Specify targets expressed as area-based units.
For example, this function can be used to express targets as
hectares, acres, or \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}.
To ensure feasibility, area-based targets are clamped based on the
total abundance of features.
}
\section{Mathematical formulation}{

This method provides an approach for setting target thresholds based
on an area-based threshold.
To express this mathematically, we will define the following terminology.
Let \eqn{f} denote the total spatial extent of a feature (e.g., geographic
range size expressed as \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}}), and
\eqn{a} the specified area-based target
(expressed as \ifelse{html}{\out{km<sup>2</sup>}}{\eqn{km^2}},
per \code{targets} and \code{area_units}).
Given this terminology, the target threshold (\eqn{t}) for the feature
is calculated as follows.
\deqn{
t = min(f, a)
}{
t = min(f, a)
}
}

\section{Data calculations}{

This function involves calculating targets based on the spatial extent
of the features in \code{x}.
Although it can be readily applied to \code{\link[=problem]{problem()}} objects that
have the feature data provided as a \code{\link[terra:rast]{terra::rast()}} object,
you will need to specify the spatial units for the features
when initializing the \code{\link[=problem]{problem()}} objects if the feature data
are provided in a different format. In particular, if the feature
data are provided as a \code{data.frame} or \code{character} vector,
then you will need to specify an argument to \code{feature_units} when
using the \code{\link[=problem]{problem()}} function.
See the Examples section of the documentation for \code{\link[=add_auto_targets]{add_auto_targets()}}
for a demonstration of specifying the spatial units for features.
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(500)

# load data
sim_complex_pu_raster <- get_sim_complex_pu_raster()
sim_complex_features <- get_sim_complex_features()

# create base problem
p0 <-
  problem(sim_complex_pu_raster, sim_complex_features) \%>\%
  add_min_set_objective() \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# create problem with targets of 50 km^2 for each feature
p1 <-
  p0 \%>\%
  add_auto_targets(
    method = spec_area_targets(targets = 50, area_units = "km^2")
  )

# solve problem
s1 <- solve(p1)

# plot solution
plot(s1, main = "solution based on constant targets", axes = FALSE)

# targets can also be specified for each feature separately.
# to demonstrate this, we will set a target value for each
# feature based on a random number between 5000 and 30000 hectares
target_values <- runif(terra::nlyr(sim_complex_features), 5000, 30000)

# create problem with targets defined separately for each feature
p2 <-
  p0 \%>\%
  add_auto_targets(
    method = spec_area_targets(targets = target_values, area_units = "ha")
  )

# solve problem
s2 <- solve(p2)

# plot solution
plot(s2, main = "solution based on varying targets", axes = FALSE)
}
}
\seealso{
Other target setting methods:
\code{\link{spec_absolute_targets}()},
\code{\link{spec_duran_targets}()},
\code{\link{spec_interp_absolute_targets}()},
\code{\link{spec_interp_area_targets}()},
\code{\link{spec_jung_targets}()},
\code{\link{spec_max_targets}()},
\code{\link{spec_min_targets}()},
\code{\link{spec_polak_targets}()},
\code{\link{spec_pop_size_targets}()},
\code{\link{spec_relative_targets}()},
\code{\link{spec_rl_ecosystem_targets}()},
\code{\link{spec_rl_species_targets}()},
\code{\link{spec_rodrigues_targets}()},
\code{\link{spec_rule_targets}()},
\code{\link{spec_ward_targets}()},
\code{\link{spec_watson_targets}()},
\code{\link{spec_wilson_targets}()}
}
\concept{methods}
